//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.3

import wesual.data.Core          1.0
import wesual.Controls           1.0

/*!
  This DropZone should be instanced inside of a SelectionGridView
  */
DropZone {
    id : dropZone

    property alias       type : handler.type

    //! provide EditContext from outside because Editor.context can't be used in
    //! showcase sidebar items
    property EditContext context
    //! dropIndicator for drop position visualization
    property Item        dropIndicator
    //! possible second dropIndicator if gridView starts new row at drop position
    property Item        dropIndicator2
    //! the target gridview of the drop action
    property GridView    gridView
    //! flag if the dropHandler should clear and add selection at drop
    property bool        handleSelection : true

    signal documentDropped(Document document)

    DocumentDropHandler {
        id : handler

         function contentPosition(viewPos) {
             return dropZone.mapToItem(
                         gridView.contentItem, viewPos.x, viewPos.y);
         }

         function rowForContentPosition(contentPos) {
             return Math.floor((contentPos.y - gridView.originY)
                               / gridView.cellHeight);
         }

         function columnForContentPosition(contentPos) {
             return Math.floor((contentPos.x - gridView.originX)
                               / gridView.cellWidth);
         }

         function cellIndex(contentPos) {
             var row = rowForContentPosition(contentPos)
             var col = columnForContentPosition(contentPos);

             var index = row * gridView.cellsPerRow + col;

             var cellPos = (contentPos.x - gridView.originX)
                     - col * gridView.cellWidth;
             if (cellPos > gridView.cellWidth / 2) {
                 index += 1;
             }

             return Math.min(index, gridView.count);
         }

         function cellPosition(index) {
             var row = Math.floor(index / gridView.cellsPerRow);
             var col = index % gridView.cellsPerRow;

             return Qt.point(col * gridView.cellWidth + gridView.originX,
                             row * gridView.cellHeight + gridView.originY);
         }

         function positionIndicator(cellIndex, contentPos) {
             if (dropIndicator2)
                 dropIndicator2.visible = false;

             var pos = cellPosition(cellIndex);
             pos = gridView.mapFromItem(
                         gridView.contentItem, pos.x, pos.y);

             if (cellIndex % gridView.cellsPerRow === 0) {
                 // Corner case: Position indicator according to row
                 var visualRow = rowForContentPosition(contentPos);
                 var row = Math.floor(cellIndex / gridView.cellsPerRow);
                 if (visualRow !== row) {
                     if (dropIndicator2) {
                         dropIndicator2.x = pos.x;
                         dropIndicator2.y = pos.y;
                         dropIndicator2.visible = true;
                     }
                     pos = cellPosition(cellIndex - 1);
                     pos.x += gridView.cellWidth;
                 }
             }

             dropIndicator.x = pos.x;
             dropIndicator.y = pos.y;
             dropIndicator.visible = true;
         }

         onEntered : {
            // for now nothing to handle
         }

         onPositionChanged : {
             if (!gridView.model || !gridView.model.isModifyable
                     || !dropIndicator)
                 return;

             gridView.scrollAnimation.stop();

             var pos = contentPosition(drag.position);
             var index = cellIndex(pos);

             // Position indicator
             positionIndicator(index, pos);
         }

         onExited : {
             if (!gridView.model.isModifyable)
                 return;

             gridView.scrollAnimation.stop();

            if (dropIndicator)
                dropIndicator.visible = false;

             if (dropIndicator2)
                 dropIndicator2.visible = false;
         }

         onDropped : {
             if (!gridView.model.isModifyable)
                 return;

             gridView.scrollAnimation.stop();
             var docs = drag.source.dragDocuments;

             var pos = contentPosition(drag.position);
             var index = cellIndex(pos);

             if (dropIndicator)
                dropIndicator.visible = false;

             if (dropIndicator2)
                 dropIndicator2.visible = false;

             if (gridView.model.documents.length > 0 && gridView.activeFocus) {
                 // check if drop index lies in selected area
                 var range = [ gridView.previousIndex, gridView.currentIndex ];

                 range.sort();
                 range[1] = range[1] + 1;

                 if (index >= range[0] && index <= range[1])
                     return;
             }

             if (handleSelection) {
                 gridView.selection = [];
             }

             var drop = true;

             if (context)
                context.beginRecordChanges();

             var newSelection = [];
             for (var i = docs.length-1; i >= 0; --i) {
                 drop = true;
                 var doc = docs[i];
                 var currentIndex = gridView.model.indexOf(doc);
                 if (currentIndex !== -1) {
                     if (currentIndex < index) {
                         index = Math.max(0, index - 1);
                     }

                     if (index === currentIndex)
                         drop = false;
                 }

                 if (drop) {
                     gridView.model.insertElement(doc, index);
                     newSelection = [doc].concat(newSelection);
                     documentDropped(doc);

                 }
             }
             if (handleSelection) {
                 gridView.selection = newSelection;
             }

             if (context)
                context.endRecordChanges();

             gridView.forceActiveFocus();
             drag.source.dragDocuments = null; // reset
         }
    }
}

